/** @file   alien.h
 * @brief   Declaration of Alien - class.
 * @version $Revision: 1.1.1.1 $
 * @date    $Date: 2006/01/21 23:02:41 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_ALIEN_H
#define H_WWW_ALIEN_H

#include "gameobject.h"


namespace WeWantWar {


/** @class  Alien
 * @brief   A baseclass of all the alien classes.
 * @author  Tomi Lamminsaari
 *
 * This class provides default implementation to handle bullethits and
 * redrawing of the aliens.
 */
class Alien : public GameObject
{
public:

  ///
  /// Constructor, destructor
  /// =======================
  
  /** Index of the counter we use for timing the reloading times. */
  static const int RELOAD_COUNTER_INDEX = 0;
  
  
  /** Constructor
   */
  Alien( );
  
  /** Destructor.
   */
  virtual ~Alien();
  
  
  
  ///
  /// Methods
  /// =======
  
  /** This alien got hit by given bullet. The bullets can go through the
   * Alien if it was shot by another alien.
   * @param     pB                Pointer to bullet that hit us.
   * @return    'true' if this bullet did hit us. 'false' means it went
   *            through.
   */
  virtual bool hitByBullet( Bullet* pB );
  
  /** Redraws this Alien. If the graphics vector has not been set, this
   * method will crash. Use @c setGraphics - method to set the graphics.
   * @param     pQueue            Pointer to redraw queue
   */
  virtual void redraw( RedrawQueue* pQueue );
  
  /** Sets the graphics vector.
   * @param     pGfx              Pointer to graphics vector. This should point
   *                              to one of the graphics vectors defined
   *                              in @c GfxPool - class.
   */
  void setGraphics( const std::vector< std::vector<RLE_SPRITE*>* >* pGfx );
  
  
  
  
  ///
  /// Getter methods
  /// ==============
  
  /** Tells if this object is loading reloading its weapon
   * @return    <code>true</code> if reloading.
   */
  virtual bool reloading() const;
  
  /** Returns the redrawing priority. This default implementation returns
   * either @c RedrawQueue::PRI_KILLED or @c RedrawQueue::PRI_NORMAL
   * @return    The redraw priority
   */
  virtual int redrawPriority() const;
  
  
protected:

  /** Sets the requested animation, unless it's already running. The
   * GameObject - class has @c setAnimation - method, but that sets the
   * given instance of Animation running. This method should check the
   * requested animation id and set the correct animation.
   * @param     aAnimId           Id of the animation such as GameAnims::WALK
   */
  virtual void setCorrectAnimation( int aAnimId ) = 0;
  
  /** Attacks. If this alien has weapon, this method should spawn the bullets.
   * The default implementation of @c update() - method calls this method
   * when the controller's shoot - flag is on.
   *
   * If you have reimplemented the update() - method, you should call this
   * method always when attacking/shooting.
   *
   * This method should respect the reloading - delays. So it should call
   * the @c reloading() - method and do nothing if we're reloading.
   */
  virtual void attack() = 0;
  
  
  
  ///
  /// Members
  /// =======
  
  /** Pointer to the vector that holds the graphics for this alien. */
  const std::vector< std::vector<RLE_SPRITE*>* >* m_pGfxTable;
  
  
private:
  /** Disabled copy constructor.
   */
  Alien( const Alien& rO );
  /** Disabled assignment operator
   */
  Alien& operator = ( const Alien& rO );
};

};  // end of namespace

#endif // ALIEN_H

/**
 * Version history
 * ===============
 * $Log: alien.h,v $
 * Revision 1.1.1.1  2006/01/21 23:02:41  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:16:46+02  lamminsa
 * Initial revision
 *
 */
